-- Copyright 2024 by Todd Hundersmarck (ThundR)
-- All Rights Reserved

_G.THGameVersion = 25
if g_minModDescVersion ~= nil then
    if g_minModDescVersion < 90 then
        _G.THGameVersion = 22
    end
end
_G.g_thGlobalEnv = _G
local meta = getmetatable(_G)
if meta ~= nil then
    if type(meta.__index) == "table" then
        _G.g_thGlobalEnv = meta.__index
    else
        printf("ERROR: Could not find global environment")
    end
end
_G.THValueType = {
    UNKNOWN = "unknown",
    STRING = "string",
    INTEGER = "integer",
    NUMBER = "number",
    BOOLEAN = "boolean",
    TABLE = "table",
    FUNCTION = "function"
}
_G.THClassType = {
    OBJECT = "object",
    EVENT = "event"
}
_G.THDebugLevel = {
    CRITICAL = 1,
    ERROR = 2,
    WARNING = 3,
    NORMAL = 4,
    UPDATE = 5,
    LOOP = 6
}
_G.THDebugLevelId = {}
for id, index in pairs(THDebugLevel) do
    THDebugLevelId[index] = id
end
local thDebugFlagIds = {}
local thGlobalDebugFlag = false
local thCurrentDebugLevel = THDebugLevel.NORMAL
THUtils = {
    modName = g_currentModName,
    modPath = g_currentModDirectory
}
THMessage = {
    INTERNAL_ERROR = "An internal error has occurred",
    ARGUMENT_INVALID = "Invalid argument %q (%s) in function call",
    FILE_NOT_FOUND = "File not found: %s",
    INVALID_FILL_TYPE = "Invalid fillType: %s",
    INVALID_FRUIT_TYPE = "Invalid fruitType: %s",
    INVALID_PARENT = "Invalid parent: %s",
    PRELOADING = "Pre-loading: %s",
    LOADING = "Loading: %s",
    DUPLICATE_ENTRY = "Duplicate %s: %s",
    INVALID_VALUE = "Invalid %s: %s",
    VALUE_OUT_OF_RANGE = "Value %q (%s) must be between %s and %s",
    VALUE_GREATER = "Value %q (%s) must be greater than %s",
    VALUE_GREATER_EQUAL = "Value %q (%s) must be greater or equal to %s",
    VALUE_LESSER = "Value %q (%s) must be less than %s",
    VALUE_LESSER_EQUAL = "Value %q (%s) must be less than or equal to %s",
    EVENT_ADMIN_ONLY = "Event requires admin privileges",
    EVENT_NO_CLINET = "Event should not run on client machines!",
    EVENT_NO_CLIENT_READ = "Event should not read from client machines!",
    EVENT_NO_CLIENT_WRITE = "Event should not write from client machines!",
    EVENT_NO_SERVER = "Event should not run on the server!",
    EVENT_NO_SERVER_READ = "Event should not read from the server!",
    EVENT_NO_SERVER_WRITE = "Event should not write from the server!"
}
function THUtils.displayMsg(text, ...)
    if text == "" then
        print(text)
    else
        local isError = false
        if type(text) == "string" then
            if select("#", ...) > 0 then
                text = string.format(text, ...)
            end
        else
            text = "ERROR: " .. string.format(THMessage.ARGUMENT_INVALID, "text", text)
            isError = true
        end
        local msgText = string.format("** [%s]: %s", THUtils.modName, text)
        print(msgText)
        if isError then
            printCallstack()
        end
    end
end
function THUtils.errorMsg(showStack, text, ...)
    if type(text) == "string" and text ~= "" then
        if showStack == nil or showStack == true then
            text = "ERROR: " .. text
        elseif showStack == false then
            text = "WARNING: " .. text
        end
    end
    THUtils.displayMsg(text, ...)
    if showStack == true then
        printCallstack()
    elseif showStack ~= nil and showStack ~= false then
        text = string.format(THMessage.ARGUMENT_INVALID, "showStack", showStack)
        THUtils.displayMsg("ERROR: " .. text)
        printCallstack()
    end
end
function THUtils.msgOnTrue(expression, showStack, text, ...)
    if expression then
        if showStack ~= nil then
            THUtils.errorMsg(showStack, text, ...)
        else
            THUtils.displayMsg(text, ...)
        end
        return true
    end
    return false
end
function THUtils.assert(expression, showStack, text, ...)
    if not expression then
        if showStack ~= nil then
            THUtils.errorMsg(showStack, text, ...)
        else
            THUtils.displayMsg(text, ...)
        end
    end
    return expression
end
function THUtils.assertValue(expression, showStack, yesValue, noValue, text, ...)
    if THUtils.assert(expression, showStack, text, ...) then
        return yesValue
    end
    return noValue
end
function THUtils.argIsValid(expression, argName, argValue)
    if not expression then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, argName, argValue)
        return false
    end
    return true
end
function THUtils.validateArg(expression, argName, argValue, defValue)
    if THUtils.argIsValid(expression, argName, argValue) then
        return argValue
    end
    return defValue
end
function THUtils.xmlErrorMsg(xmlKey, showStack, text, ...)
    text = text or ""
    if THUtils.argIsValid(type(text) == "string", "text", text) then
        local xmlText = "XML Warning: %s"
        if showStack == nil or showStack == true then
            xmlText = "XML Error: %s"
        elseif showStack ~= false then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "showStack", showStack)
        end
        THUtils.displayMsg(xmlText, xmlKey)
        if text ~= nil and text ~= "" then
            THUtils.displayMsg("- " .. text, ...)
        end
        if showStack == true then
            printCallstack()
        end
    end
end
function THUtils.getDebugFlagId(flagId)
    if flagId == nil then
        flagId = ""
    elseif type(flagId) == "string" then
        if flagId ~= "" then
            flagId = flagId:upper()
        end
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "flagId", flagId)
        return
    end
    if flagId == "" or thDebugFlagIds[flagId] ~= nil then
        return flagId
    end
    THUtils.errorMsg(nil, "Debug flag %q does not exist")
end
function THUtils.getDebugFlagIds()
    return thDebugFlagIds
end
function THUtils.getDebugLevel()
    return thCurrentDebugLevel
end
function THUtils.getDebugLevelById(debugLevelId)
    local idType = type(debugLevelId)
    local debugLevel = nil
    if idType == "string" then
        debugLevel = THDebugLevel[debugLevelId:upper()]
        if debugLevel == nil then
            debugLevelId = THUtils.toNumber(debugLevelId, true)
            if debugLevelId ~= nil then
                debugLevel = THDebugLevelId[debugLevelId]
            end
        end
    elseif idType == "number" then
        debugLevel = THDebugLevelId[debugLevelId]
    else
        THUtils.errorMsg(nil, "Invalid debug level: %q", debugLevelId)
    end
    return debugLevel
end
function THUtils.setDebugLevel(debugLevelId)
    local debugLevel = THUtils.getDebugLevelById(debugLevelId)
    local success = false
    if debugLevel ~= nil then
        thCurrentDebugLevel = debugLevel
        success = true
    end
    return success, thCurrentDebugLevel
end
function THUtils.createDebugFlag(flagName)
    if THUtils.argIsValid(THUtils.validateId(flagName), "flagName", flagName) then
        local flagId = flagName:upper()
        if thDebugFlagIds[flagId] == nil then
            thDebugFlagIds[flagId] = false
        end
        return flagId
    end
end
function THUtils.getIsDebugEnabled(flagName, debugLevelId)
    local reqDebugLevel = nil
    if debugLevelId == nil then
        reqDebugLevel = thCurrentDebugLevel
    else
        reqDebugLevel = THUtils.getDebugLevelById(debugLevelId)
    end
    local flagId = THUtils.getDebugFlagId(flagName)
    local isDebugEnabled = false
    if flagId ~= nil
        and reqDebugLevel ~= nil and reqDebugLevel <= thCurrentDebugLevel
    then
        if flagId == "" then
            isDebugEnabled = thGlobalDebugFlag == true
        else
            isDebugEnabled = thDebugFlagIds[flagId] == true
        end
    end
    return isDebugEnabled, flagId
end
function THUtils.setIsDebugEnabled(flagName, isEnabled)
    local flagId = THUtils.getDebugFlagId(flagName)
    if flagId ~= nil then
        isEnabled = THUtils.getNoNil(isEnabled, true)
        if THUtils.argIsValid(not isEnabled or isEnabled == true, "isEnabled", isEnabled) then
            if flagId == "" then
                thGlobalDebugFlag = isEnabled == true
                return true, thGlobalDebugFlag, flagId
            else
                thDebugFlagIds[flagId] = isEnabled == true
                return true, thDebugFlagIds[flagId], flagId
            end
        end
    end
    return false
end
function THUtils.resetDebug()
    THUtils.setIsDebugEnabled(nil, false)
    THUtils.setDebugLevel(THDebugLevel.NORMAL)
end
function THUtils.printTable(target, ...)
    if g_thDebugTools ~= nil then
        return g_thDebugTools:printTable(target, ...)
    end
    local targetName = target
    if type(target) == "string" then
        target = THUtils.getTableValue(nil, targetName)
    end
    if THUtils.argIsValid(type(target) == "table", "target", targetName) then
        THUtils.displayMsg("Printing %s")
        for key, value in pairs(target) do
            THUtils.displayMsg("[%s]: %s", key, value)
        end
    end
end
THUtils.dumpTable = THUtils.printTable
function THUtils.traceFunction(...)
    if g_thDebugTools ~= nil then
        return g_thDebugTools:traceFunction(...)
    end
end
function THUtils.debugMsg(flagName, debugLevelId, text, ...)
    if THUtils.getIsDebugEnabled(flagName, debugLevelId) then
        local debugLevel = THUtils.getDebugLevelById(debugLevelId)
        if debugLevel == THDebugLevel.CRITICAL then
            THUtils.errorMsg(true, text, ...)
        elseif debugLevel == THDebugLevel.ERROR then
            THUtils.errorMsg(nil, text, ...)
        elseif debugLevel == THDebugLevel.WARNING then
            THUtils.errorMsg(false, text, ...)
        else
            THUtils.displayMsg(text, ...)
        end
    end
end
function THUtils.pack(...)
    return { n = select("#", ...), ... }
end
function THUtils.unpack(target, index, endIndex)
    if THUtils.argIsValid(type(target) == "table", "target", target) then
        if index ~= nil and type(index) ~= "number" then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "index", index)
            index = 1
        end
        if type(endIndex) ~= "number" then
            if endIndex ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "endIndex", endIndex)
            end
            endIndex = target.n
        end
        return unpack(target, index, endIndex)
    end
end
function THUtils.getNoNil(value, replacement)
    if value == nil then
        return replacement
    end
    return value
end
function THUtils.evaluate(expression, yesValue, noValue)
    if expression then
        return yesValue
    end
    return noValue
end
function THUtils.pcall(func, pa,pb,pc,pd,pe,pf,pg,ph,pi,pj,pk,pl,pm,pn,po,pp,pq,pr,ps,pt)
    local function errHandler(errMsg)
        errMsg = THUtils.getNoNil(errMsg, THMessage.INTERNAL_ERROR)
        if type(errMsg) == "string" then
            THUtils.errorMsg(true, errMsg)
        else
            printCallstack()
        end
    end
    local function protectedFunc()
        return func(pa,pb,pc,pd,pe,pf,pg,ph,pi,pj,pk,pl,pm,pn,po,pp,pq,pr,ps,pt)
    end
    return xpcall(protectedFunc, errHandler)
end
function THUtils.call(func, ...)
    local function appendFunc(rSuccess, ret1, ...)
        if rSuccess == true then
            return ret1, ...
        end
    end
    return appendFunc(THUtils.pcall(func, ...))
end
function THUtils.getFilename(filename, filePath, verbose, ...)
    if THUtils.argIsValid(filename == nil or type(filename) == "string", "filename", filename)
        and THUtils.argIsValid(filePath == nil or type(filePath) == "string", "filePath", filePath)
        and THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose)
    then
        if filename ~= nil then
            local function appendFunc(rAbsFilename, ...)
                THUtils.call(function()
                    if verbose == true then
                        if rAbsFilename == nil or not fileExists(rAbsFilename) then
                            THUtils.errorMsg(true, THMessage.FILE_NOT_FOUND, rAbsFilename)
                        end
                    end
                end)
                return rAbsFilename, ...
            end
            return appendFunc(Utils.getFilename(filename, filePath, ...))
        end
    end
    return ""
end
function THUtils.toBoolean(value)
    if value == true or value == false then
        return value
    elseif type(value) == "string" then
        local upperVal = value:upper()
        if upperVal == "TRUE" then
            return true
        elseif upperVal == "FALSE" then
            return false
        end
    end
end
function THUtils.toNumber(value, isInteger)
    if value ~= nil then
        value = tonumber(value)
        if isInteger == true then
            if value ~= nil then
                value = math.floor(value)
            end
        elseif isInteger ~= nil and isInteger ~= false then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "isInteger", isInteger)
        end
        return value
    end
end
function THUtils.getIsType(value, reqType)
    local valType = type(value)
    if valType == reqType then
        return true
    end
    if type(reqType) == "table" then
        if valType == "table" and value.isa ~= nil and value:isa(reqType) then
            return true
        end
    elseif reqType == "integer" then
        if valType == "number" and math.floor(value) == value then
            return true
        end
    end
    return false
end
function THUtils.convertValue(value, valueTypeId)
    local valueType, newValue = nil, nil
    if type(valueTypeId) == "string" then
        valueType = THValueType[valueTypeId:upper()]
    end
    if THUtils.argIsValid(valueType ~= nil, "valueTypeId", valueTypeId) then
        if valueType == THValueType.STRING then
            newValue = tostring(value)
        elseif valueType == THValueType.INTEGER then
            newValue = THUtils.toNumber(value, true)
        elseif valueType == THValueType.NUMBER then
            newValue = THUtils.toNumber(value)
        elseif valueType == THValueType.BOOLEAN then
            newValue = THUtils.toBoolean(value)
        else
            THUtils.errorMsg(false, "Values of type %q cannot be converted", valueType)
        end
    end
    return newValue
end
function THUtils.floor(value, precision)
    value = THUtils.validateArg(type(value) == "number", "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local factor = 10 ^ precision
    return math.floor(value * factor) / factor
end
function THUtils.ceil(value, precision)
    value = THUtils.validateArg(type(value) == "number", "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local factor = 10 ^ precision
    return math.ceil(value * factor) / factor
end
function THUtils.round(value, precision)
    value = THUtils.validateArg(type(value) == "number", "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local factor = 10 ^ precision
    return math.floor((value * factor) + 0.5) / factor
end
function THUtils.getVector2Distance(sx,sz, ex,ez)
    if THUtils.argIsValid(type(sx) == "number", "sx", sx)
        and THUtils.argIsValid(type(sz) == "number", "sz", sz)
        and THUtils.argIsValid(type(ex) == "number", "ex", ex)
        and THUtils.argIsValid(type(ez) == "number", "ez", ez)
    then
        return math.sqrt(((ex - sx) ^ 2) + ((ez - sz) ^ 2))
    end
    return 0
end
function THUtils.getVector3Distance(sx,sy,sz, ex,ey,ez)
    if THUtils.argIsValid(type(sx) == "number", "sx", sx)
        and THUtils.argIsValid(type(sy) == "number", "sy", sy)
        and THUtils.argIsValid(type(sz) == "number", "sz", sz)
        and THUtils.argIsValid(type(ex) == "number", "ex", ex)
        and THUtils.argIsValid(type(ey) == "number", "ey", ey)
        and THUtils.argIsValid(type(ez) == "number", "ez", ez)
    then
        return math.sqrt(((ex - sx) ^ 2) + ((ey - sy) ^ 2) + ((ez - sz) ^ 2))
    end
    return 0
end
function THUtils.getVector2AreaByLine(sx, sz, ex, ez, width)
    if THUtils.argIsValid(type(sx) == "number", "sx", sx)
        and THUtils.argIsValid(type(sz) == "number", "sz", sz)
        and THUtils.argIsValid(type(ex) == "number", "ex", ex)
        and THUtils.argIsValid(type(ez) == "number", "ez", ez)
    then
        if type(width) ~= "number" or width < 0 then
            if width ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "width", width)
            end
            width = 0
        end
        local length = THUtils.getVector2Distance(sx, sz, ex, ez)
        local mx = (sx + ex) / 2
        local mz = (sz + ez) / 2
        if length == 0 then -- single point
            if width == 0 then -- return single point
                return mx, mz, mx, mz, mx, mz
            end
            local halfWidth = width / 2
            return mx - halfWidth, mz - halfWidth, mx + halfWidth, mz - halfWidth, mx - halfWidth, mz + halfWidth
        end
        local factor = 1 + (width / length)
        local dx = ((ex - sx) / 2) * factor
        local dz = ((ez - sz) / 2) * factor
        sx = mx - dx
        sz = mz - dz
        ex = mx + dx
        ez = mz + dz
        local rdx = dz
        local rdz = -dx
        return sx + rdx, sz + rdz, sx - rdx, sz - rdz, ex + rdx, ez + rdz
    end
end
function THUtils.getFactorInRange(val1, val2, testVal, noClamp)
    if THUtils.argIsValid(type(val1) == "number", "val1", val1)
        and THUtils.argIsValid(type(val2) == "number", "val2", val2)
        and THUtils.argIsValid(type(testVal) == "number", "testVal", testVal)
    then
        if val1 == val2 then
            if testVal < val1 then
                return 0
            else
                return 1
            end
        end
        local factor = (testVal - val1) / (val2 - val1)
        if noClamp ~= true then
            if noClamp ~= nil and noClamp ~= false then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "noClamp", noClamp)
            end
            factor = MathUtil.clamp(factor, 0, 1)
        end
        return factor
    end
    return 0
end
function THUtils.getValueInRange(val1, val2, alpha, noClamp)
    if THUtils.argIsValid(type(val1) == "number", "val1", val1)
        and THUtils.argIsValid(type(val2) == "number", "val2", val2)
        and THUtils.argIsValid(type(alpha) == "number", "alpha", alpha)
    then
        local newValue = val1 + ((val2 - val1) * alpha)
        if noClamp ~= true then
            if noClamp ~= nil and noClamp ~= false then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "noClamp", noClamp)
            end
            local minVal = math.min(val1, val2)
            local maxVal = math.max(val1, val2)
            newValue = MathUtil.clamp(newValue, minVal, maxVal)
        end
        return newValue
    end
    if type(val1) == "number" then
        return val1
    elseif type(val2) == "number" then
        return val2
    end
    return 0
end
function THUtils.splitString(str, pattern, isList)
    if type(pattern) ~= "string" or pattern == "" then
        if pattern ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "pattern", pattern)
        end
        pattern = " "
    end
    local splitTable = {}
    if str == nil or str == "" then
        return splitTable
    end
    if THUtils.argIsValid(type(str) == "string", "str", str)
        and THUtils.argIsValid(not isList or isList == true, "isList", isList)
    then
        local valToAdd, valLength = nil, string.len(str)
        local strPos = 1
        while true do
            if strPos <= 0 or strPos > valLength then
                break
            end
            local strStart, strEnd = string.find(str, pattern, strPos, true)
            if strStart == nil or strEnd == nil then
                valToAdd = string.sub(str, strPos)
                strPos = valLength + 1
            else
                if strEnd <= 1 then
                    valToAdd = nil
                else
                    valToAdd = string.sub(str, strPos, strEnd - 1)
                end
                strPos = strEnd + 1
            end
            if valToAdd ~= nil and valToAdd ~= "" and valToAdd ~= pattern then
                if isList == true then
                    splitTable[valToAdd] = true
                else
                    table.insert(splitTable, valToAdd)
                end
            end
        end
    end
    return splitTable
end
function THUtils.properCase(str, noSpace)
    if THUtils.argIsValid(type(str) == "string", "str", str) then
        local newStr = string.gsub(str, "^%l", string.upper)
        newStr = string.gsub(newStr, " %l", string.upper)
        if noSpace == true then
            newStr = string.gsub(newStr, " ", "")
        elseif noSpace ~= nil and noSpace ~= false then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "noSpace", noSpace)
        end
        return newStr
    end
    return str
end
function THUtils.snakeCase(str, toUpper)
    if THUtils.argIsValid(type(str) == "string", "str", str) then
        local newStr = string.gsub(str, " ", "_")
        newStr = string.gsub(newStr, "(%l)(%u)", "%1_%2")
        if toUpper == true then
            newStr = newStr:upper()
        else
            if toUpper ~= false and toUpper ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "toUpper", toUpper)
                toUpper = false
            end
            newStr = newStr:lower()
        end
        return newStr
    end
    return str
end
function THUtils.validateId(id)
    if id ~= nil and id ~= "" and type(id) == "string"
        and not string.find(id, "[^A-Za-z0-9_]")
    then
        return true
    end
    return false
end
function THUtils.formatPercent(value, precision)
    value = THUtils.validateArg(type(value) == "number", "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local percentValue = THUtils.round(value * 100, precision)
    local percentText = string.format("%0." .. tostring(precision) .. "f%%", percentValue)
    return percentText
end
function THUtils.clearTable(target)
    if THUtils.argIsValid(type(target) == "table", "target", target) then
        for key in pairs(target) do
            target[key] = nil
        end
    end
    return target
end
function THUtils.getTableValue(target, targetPath, pattern)
    if target == nil then
        target = g_thGlobalEnv
    elseif type(target) ~= "table" then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "target", target)
        return
    end
    if type(pattern) ~= "string" or pattern == "" then
        if pattern ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "pattern", pattern)
        end
        pattern = "."
    end
    if targetPath == nil or targetPath == "" then
        return target
    end
    if THUtils.argIsValid(type(targetPath) == "string", "targetPath", targetPath) then
        if not string.find(targetPath, pattern, nil, true) then
            return target[targetPath]
        end
        local tableKeys = THUtils.splitString(targetPath, pattern)
        local numEntries = tableKeys ~= nil and #tableKeys
        if numEntries > 0 then
            local finalValue = target
            for entryIndex = 1, numEntries do
                local entry = tableKeys[entryIndex]
                if entryIndex > 1 and type(finalValue) ~= "table" then
                    THUtils.errorMsg(true, "Invalid table: %s", entry)
                    return
                end
                if finalValue[entry] ~= nil then
                    finalValue = finalValue[entry]
                else
                    local otherEntry = THUtils.toBoolean(entry)
                    if otherEntry ~= nil and finalValue[otherEntry] ~= nil then
                        finalValue = finalValue[otherEntry]
                    else
                        otherEntry = THUtils.toNumber(entry)
                        if otherEntry ~= nil and finalValue[otherEntry] ~= nil then
                            finalValue = finalValue[otherEntry]
                        end
                    end
                end
            end
            return finalValue
        end
    end
end
function THUtils.copyTable(target, depth, copyMeta)
    local newTable = {}
    if THUtils.argIsValid(type(target) == "table", "target", target) then
        if type(depth) ~= "number" or depth < 0 then
            if depth ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "depth", depth)
            end
            depth = 0
        else
            depth = math.floor(depth)
        end
        for key, val in pairs(target) do
            if type(val) == "table" and depth > 0 then
                newTable[key] = THUtils.copyTable(val, depth - 1)
            else
                newTable[key] = val
            end
        end
        if copyMeta == true then
            local oldMt = getmetatable(target)
            if oldMt ~= nil then
                setmetatable(newTable, oldMt)
            end
        elseif copyMeta ~= nil and copyMeta ~= false then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "copyMeta", copyMeta)
        end
    end
    return newTable
end
function THUtils.createSubTable(parent, tableKey, clear, useRaw)
    if THUtils.argIsValid(type(parent) == "table", "parent", parent)
        and THUtils.argIsValid(tableKey ~= nil, "tableKey", tableKey)
    then
        local newTable = nil
        if useRaw == true then
            newTable = rawget(parent, tableKey)
        else
            newTable = parent[tableKey]
        end
        if newTable == nil then
            newTable = {}
            if useRaw == true then
                rawset(parent, tableKey, newTable)
            else
                parent[tableKey] = newTable
            end
        else
            if clear == true then
                THUtils.clearTable(newTable)
            elseif clear ~= nil and clear ~= false then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "clear", clear)
            end
        end
        return newTable
    end
end
function THUtils.getTableLength(target)
    if THUtils.argIsValid(type(target) == "table", "target", target) then
        local numEntries = 0
        if target ~= nil then
            for _ in pairs(target) do
                numEntries = numEntries + 1
            end
        end
        return numEntries
    end
    return 0
end
function THUtils.getDataTable(target, dataKey)
    if type(target) == "table" then
        dataKey = THUtils.getNoNil(dataKey, target)
        local dataTable = rawget(target, dataKey)
        return dataTable, target
    end
    if target ~= nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "target", target)
    end
end
function THUtils.createDataTable(target, dataKey, dataClass, ...)
    if THUtils.argIsValid(type(target) == "table", "target", target) then
        dataKey = THUtils.getNoNil(dataKey, target)
        local dataTable = THUtils.getDataTable(target, dataKey)
        if dataTable == nil then
            if type(dataClass) == "table" then
                if dataClass.newData ~= nil then
                    dataTable = dataClass.newData(target, dataKey, ...)
                else
                    dataTable = setmetatable({}, { __index = dataClass })
                end
            else
                if dataClass ~= nil then
                    THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "dataClass", dataClass)
                end
                dataTable = {}
            end
            if dataTable ~= nil then
                if dataTable.parent == nil then
                    dataTable.parent = target
                end
                if dataTable.owner == nil and type(dataKey) == "table" then
                    dataTable.owner = dataKey
                end
                rawset(target, dataKey, dataTable)
            end
        end
        return dataTable
    end
end
function THUtils.getSpecTable(object, specName, modName)
    if THUtils.argIsValid(type(specName) == "string", "specName", specName) then
        if type(object) ~= "table" then
            if object ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "object", object)
            end
            return
        end
        local specKey = nil
        if modName == nil then
            specKey = string.format("spec_%s", specName)
        elseif type(modName) == "string" then
            specKey = string.format("spec_%s.%s", modName, specName)
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "modName", modName)
        end
        if specKey ~= nil then
            return object[specKey], object
        end
        return nil, object
    end
end
function THUtils.makeSelfCallback(target, targetFunc)
    if target == nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "target", target)
    else
        local callbackFunc = function(...)
            return targetFunc(target, ...)
        end
        return callbackFunc
    end
end
function THUtils.addTask(isProtected, taskFunc)
    local taskManager = g_asyncTaskManager
    if taskManager == nil then
        THUtils.errorMsg(true, "Could not find task manager")
    elseif THUtils.argIsValid(type(taskFunc) == "function", "taskFunc", taskFunc) then
        if isProtected == true then
            taskManager:addTask(function()
                THUtils.pcall(taskFunc)
            end)
        else
            if isProtected ~= nil and isProtected ~= false then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "isProtected", isProtected)
            end
            taskManager:addTask(taskFunc)
        end
    end
end
function THUtils.addSubTask(isProtected, taskFunc)
    local taskManager = g_asyncTaskManager
    if taskManager == nil then
        THUtils.errorMsg(true, "Could not find task manager")
    elseif THUtils.argIsValid(type(taskFunc) == "function", "taskFunc", taskFunc) then
        if isProtected == true then
            taskManager:addSubtask(function()
                THUtils.pcall(taskFunc)
            end)
        else
            if isProtected ~= nil and isProtected ~= false then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "isProtected", isProtected)
            end
            taskManager:addSubtask(taskFunc)
        end
    end
end
function THUtils.addCallbackToArray(callbackArray, callbackTarget, callbackFunc, ...)
    if THUtils.argIsValid(type(callbackArray) == "table", "callbackArray", callbackArray)
        and THUtils.argIsValid(type(callbackFunc) == "function", "callbackFunc", callbackFunc)
    then
        local numEntries = #callbackArray
        local callbackData = nil
        local isCallbackFound = false
        if numEntries > 0 then
            for entryIndex = 1, numEntries do
                callbackData = callbackArray[entryIndex]
                if callbackData.func == callbackFunc then
                    isCallbackFound = true
                    break
                end
            end
        end
        if not isCallbackFound then
            callbackData = {
                func = callbackFunc,
                target = callbackTarget,
                args = THUtils.pack(...)
            }
            table.insert(callbackArray, callbackData)
        end
        return true, callbackFunc
    end
    return false
end
function THUtils.removeCallbackFromArray(callbackArray, callbackFunc)
    if THUtils.argIsValid(type(callbackArray) == "table", "callbackArray", callbackArray) then
        if type(callbackFunc) == "function" then
            local entryIndex = 1
            local callbackData = nil
            local isCallbackFound = false
            while true do
                callbackData = callbackArray[entryIndex]
                if callbackData == nil then
                    break
                end
                if callbackData.func == callbackFunc then
                    table.remove(callbackArray, entryIndex)
                    isCallbackFound = true
                else
                    entryIndex = entryIndex + 1
                end
            end
            return isCallbackFound
        elseif callbackFunc ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "callbackFunc", callbackFunc)
        end
    end
    return false
end
function THUtils.raiseArrayCallbacks(callbackArray, ...)
    if THUtils.argIsValid(type(callbackArray) == "table", "callbackArray", callbackArray) then
        local callbackArgs = THUtils.pack(...)
        for entryIndex = 1, #callbackArray do
            local callbackData = callbackArray[entryIndex]
            local hasCallbackArgs = callbackData.args.n > 0
            local function taskFunc()
                if callbackData.target ~= nil then
                    if hasCallbackArgs then
                        callbackData.func(callbackData.target, THUtils.unpack(callbackData.args), THUtils.unpack(callbackArgs))
                    else
                        callbackData.func(callbackData.target, THUtils.unpack(callbackArgs))
                    end
                else
                    if hasCallbackArgs then
                        callbackData.func(THUtils.unpack(callbackData.args), THUtils.unpack(callbackArgs))
                    else
                        callbackData.func(THUtils.unpack(callbackArgs))
                    end
                end
            end
            if entryIndex == 1 then
                THUtils.addTask(true, taskFunc)
            else
                THUtils.addSubTask(true, taskFunc)
            end
        end
    end
end
function THUtils.registerFunction(target, funcName, newFunc)
    if THUtils.argIsValid(type(target) == "table", "target", target)
        and THUtils.argIsValid(type(funcName) == "string", "funcName", funcName)
        and THUtils.argIsValid(type(newFunc) == "function", "newFunc", newFunc)
    then
        local oldFunc = rawget(target, funcName)
        if oldFunc ~= nil then
            THUtils.errorMsg(true, "Function %q already registered", funcName)
            return oldFunc
        end
        local callbackFunc = function(...)
            return newFunc(...)
        end
        rawset(target, funcName, callbackFunc)
        return callbackFunc
    end
end
function THUtils.setFunctionHook(srcTable, srcFuncName, allowCreate, useSelf, extraArg, tgtFunc)
    if srcTable == nil then
        srcTable = _G
    else
        local srcTableName = srcTable
        if type(srcTable) == "string" then
            srcTable = THUtils.getTableValue(nil, srcTableName)
        end
        if type(srcTable) ~= "table" then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "srcTable", srcTableName)
            return
        end
    end
    if THUtils.argIsValid(type(srcFuncName) == "string", "srcFuncName", srcFuncName)
        and THUtils.argIsValid(type(tgtFunc) == "function", "tgtFunc", tgtFunc)
        and THUtils.argIsValid(not allowCreate or allowCreate == true, "allowCreate", allowCreate)
    then
        local srcFunc = srcTable[srcFuncName]
        if type(srcFunc) ~= "function" then
            if srcFunc == nil and allowCreate == true then
                srcFunc = function() end
            else
                THUtils.errorMsg(true, "Invalid source function: %q", srcFuncName)
                return
            end
        end
        local function callbackFunc(p1, ...)
            if useSelf == true then
                if extraArg ~= nil then
                    return tgtFunc(p1, srcFunc, extraArg, ...)
                else
                    return tgtFunc(p1, srcFunc, ...)
                end
            else
                if extraArg ~= nil then
                    return tgtFunc(extraArg, srcFunc, p1, ...)
                else
                    return tgtFunc(srcFunc, p1, ...)
                end
            end
        end
        rawset(srcTable, srcFuncName, callbackFunc)
        return callbackFunc, srcTable
    end
end
function THUtils.getIsMasterUser()
    if g_currentMission ~= nil then
        return g_currentMission.isMasterUser == true
    end
    return false
end
function THUtils.getModEnvironment(modName)
    if modName == nil then
        return _G
    end
    if THUtils.argIsValid(type(modName) == "string", "modName", modName) then
        local modEnv = g_thGlobalEnv[modName]
        if modEnv ~= nil and modEnv._G ~= nil then
            return modEnv._G
        end
    end
end
function THUtils.getClassObject(className, modName)
    local classObject = nil
    if modName == nil then
        classObject = THUtils.getTableValue(g_thGlobalEnv, className)
    else
        local modEnv = THUtils.getModEnvironment(modName)
        if modEnv ~= nil then
            classObject = THUtils.getTableValue(modEnv, className)
        end
    end
    return classObject
end
function THUtils.createClass(baseClass, parentClass, ...)
    if THUtils.argIsValid(type(baseClass) == "table", "baseClass", baseClass)
        and THUtils.argIsValid(parentClass == nil or type(parentClass) == "table", "parentClass", parentClass)
    then
        return Class(baseClass, parentClass, ...)
    end
end
function THUtils.initClass(classTable, className, classType, ...)
    if THUtils.argIsValid(type(classTable) == "table", "classTable", classTable)
        and THUtils.argIsValid(type(className) == "string" and className ~= "", "className", className)
    then
        if classType == THClassType.OBJECT then
            return InitObjectClass(classTable, className, ...)
        elseif classType == THClassType.EVENT then
            return InitEventClass(classTable, className, ...)
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "classType", classType)
        end
    end
end
function THUtils.loadXMLFile(xmlName, filename, filePath, xmlSchema)
    local xmlFile = nil
    local xmlFilename = THUtils.getFilename(filename, filePath)
    if THUtils.argIsValid(type(xmlName) == "string", "xmlName", xmlName)
        and xmlFilename ~= ""
    then
        if xmlSchema == false then
            if fileExists(xmlFilename) then
                xmlFile = loadXMLFile(xmlName, xmlFilename)
                if xmlFile ~= nil and xmlFile <= 0 then
                    THUtils.errorMsg(true, "Invalid xml file (%s)", xmlName)
                    return
                end
            end
        elseif xmlSchema == nil or xmlSchema == true then
            xmlFile = XMLFile.loadIfExists(xmlName, xmlFilename)
        elseif type(xmlSchema) == "table" then
            xmlFile = XMLFile.loadIfExists(xmlName, xmlFilename, xmlSchema)
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlSchema", xmlSchema)
        end
        return xmlFile
    end
end
function THUtils.deleteXMLFile(xmlFile)
    local xmlFileType = type(xmlFile)
    if xmlFileType == "table" then
        xmlFile:delete()
    elseif xmlFileType == "number" then
        delete(xmlFile)
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlFile", xmlFile)
    end
end
function THUtils.registerXMLPath(xmlSchema, ...)
    if THUtils.argIsValid(type(xmlSchema) == "table" and xmlSchema.register ~= nil, "xmlSchema", xmlSchema) then
        return xmlSchema:register(...)
    end
end
function THUtils.hasXMLProperty(xmlFile, ...)
    if type(xmlFile) == "table" and xmlFile.hasProperty ~= nil then
        return xmlFile:hasProperty(...)
    elseif THUtils.getIsType(xmlFile, "integer") and xmlFile > 0 then
        return hasXMLProperty(xmlFile, ...)
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlFile", xmlFile)
    end
    return false
end
function THUtils.getXMLValueFunc(xmlFile, xmlValueType)
    local xmlFileType = type(xmlFile)
    local getValueFunc = nil
    local hasExtraArgs = xmlFileType == "table"
    if THUtils.argIsValid(xmlFileType == "table" or xmlFileType == "number", "xmlFile", xmlFile) then
        if xmlValueType == nil then
            if xmlFileType == "table" then
                getValueFunc = xmlFile.getValue
            end
        elseif xmlValueType == XMLValueType.STRING then
            if xmlFileType == "table" then
                getValueFunc = xmlFile.getString
            else
                getValueFunc = getXMLString
            end
        elseif xmlValueType == XMLValueType.FLOAT then
            if xmlFileType == "table" then
                getValueFunc = xmlFile.getFloat
            else
                getValueFunc = getXMLFloat
            end
        elseif xmlValueType == XMLValueType.INT then
            if xmlFileType == "table" then
                getValueFunc = xmlFile.getInt
            else
                getValueFunc = getXMLInt
            end
        elseif xmlValueType == XMLValueType.BOOL then
            if xmlFileType == "table" then
                getValueFunc = xmlFile.getBool
            else
                getValueFunc = getXMLBool
            end
        end
    end
    if getValueFunc == nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlValueType", xmlValueType)
    end
    return getValueFunc, hasExtraArgs
end
function THUtils.getXMLValue(xmlFile, xmlValueType, xmlKey, defaultValue, ...)
    local getValueFunc, hasExtraArgs = THUtils.getXMLValueFunc(xmlFile, xmlValueType)
    if getValueFunc ~= nil then
        local function appendFunc(rValue, ...)
            if rValue == nil and not hasExtraArgs then
                rValue = defaultValue
            end
            return rValue, ...
        end
        if hasExtraArgs then
            return appendFunc(getValueFunc(xmlFile, xmlKey, defaultValue, ...))
        end
        return appendFunc(getValueFunc(xmlFile, xmlKey))
    end
    return defaultValue
end
function THUtils.setXMLValue(xmlFile, ...)
    if THUtils.argIsValid(type(xmlFile) == "table" and xmlFile.setValue ~= nil, "xmlFile", xmlFile) then
        return xmlFile:setValue(...)
    end
end
function THUtils.getNormalizedValues(uiScale, x, y)
    local normX, normY = 0, 0
    if THUtils.argIsValid(type(uiScale) == "number", "uiScale", uiScale) then
        if type(x) == "number" then
            normX = x * uiScale * g_aspectScaleX / g_referenceScreenWidth
        elseif x ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "x", x)
        end
        if type(y) == "number" then
            normY = y * uiScale * g_aspectScaleY / g_referenceScreenHeight
        elseif y ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "y", y)
        end
    end
    return normX, normY
end
function THUtils.getElementByProfileName(baseElement, profileName, depth)
    if type(depth) ~= "number" or depth < 0 then
        if depth ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "depth", depth)
        end
        depth = 0
    else
        depth = math.floor(depth)
    end
    if THUtils.argIsValid(type(baseElement) == "table", "baseElement", baseElement) then
        if profileName == nil then
            return
        end
        if profileName == baseElement.profile then
            return baseElement
        end
        local elementList = baseElement.elements
        local numElements = elementList ~= nil and #elementList
        if numElements > 0 then
            for idx = 1, #elementList do
                local element = elementList[idx]
                if depth > 0 then
                    local foundElement = THUtils.getElementByProfileName(element, profileName, depth - 1)
                    if foundElement ~= nil then
                        return foundElement
                    end
                elseif profileName == element.profile then
                    return element
                end
            end
        end
    end
end